Attribute VB_Name = "modPnMProgram"
' TK3 programming function for new PIC types introduced by PIC n' Mix
' decoupled from the rest of TK3 to ease maintenance
' Written by Andrew Jarvis 26.03.04
'
' High Level TK3 hardware interface routines
' Mostly wrappers of existing JB code.
'
' Revision History
' 26.03.04 Initial Revision
' 15.08.04 Final Revision
'
Option Explicit

'PIC command codes
Private Const LOAD_CONFIGURATION = &H0
Private Const LOAD_DATA_FOR_PROGRAM_MEMORY = &H2
Private Const LOAD_DATA_FOR_DATA_MEMORY = &H3
Private Const READ_DATA_FROM_PROGRAM_MEMORY = &H4
Private Const READ_DATA_FROM_DATA_MEMORY = &H5
Private Const INCREMENT_ADDRESS = &H6
Private Const BEGIN_PROGRAMMING = &H8
Private Const BULK_ERASE_PROGRAM_MEMORY = &H9
Private Const BULK_ERASE_DATA_MEMORY = &HB
Private Const CHIP_ERASE = &H1F

Public Sub PnM_EraseProgram()
    
    Dim oscL As Byte
    Dim oscH As Byte
    
    PnM_EnterProgramMode
    
    'save configuration memory if necessary. Note no need to
    'store BG as well because it won't get erased here
    If PnM_LowPinCount Then
        PnM_ReadOscCal PICsize - 1, oscL, oscH
    End If
    
    delay10
    PnM_Erase
    
    'put things back as we expect to find them
    If PnM_LowPinCount Then
        PnM_RestoreOscCal PICsize - 1, oscL, oscH
    End If

    PnM_ExitProgramMode
            
End Sub
Public Sub PnM_Erase()
    
    BulkEraseProgram
    BeginProgramCycle

End Sub
Public Sub PnM_CPErase()
    
    'perform a chip erase
    If PnM_16f87xA Then
    
        PnM_EnterProgramMode
        ChipErase
        PnM_ExitProgramMode
        
    Else
    End If
    
End Sub
Public Sub PnM_EraseEEPROM()

    'erase EEPROM data memory
    PnM_EnterProgramMode
    BulkEraseEEPROM
    BeginProgramCycle
    PnM_ExitProgramMode
    
End Sub
Public Sub PnM_EnterProgramMode()
    
    'JB code to set programming mode
    Out Port1, 16
    Call delay10
    Out Port1, (8 Or 16)
    Call delay10
    Out Port1, 16
    
End Sub
Public Sub PnM_ExitProgramMode()

    'JB code to exit programming mode
   Out Port1, 8
   Call delay10
   Out Port1, 0
   
End Sub
Public Sub PnM_EnterConfigurationMode()

    'enter configuration mode - needs to be done after a reset
    PnM_ExitProgramMode
    PnM_EnterProgramMode
    SetConfiguration
    
End Sub
Public Sub PnM_WriteProgramMemory(ByVal lsb As Byte, ByVal msb As Byte, Optional ByVal beginprogramming As Boolean = True)

    Dim wordLSB As Long
    Dim wordMSB As Long
    
    'convert to longs ready for maths below
    wordLSB = lsb
    wordMSB = msb
    
    Dim outval As Long
    outval = wordMSB * 256 + wordLSB
    
    SendCommand LOAD_DATA_FOR_PROGRAM_MEMORY
    WriteBits outval * 2
    
    'some algorithms use a BeginProgramming here, others do not
    If beginprogramming Then
        BeginProgramCycle
        delay10
    End If
    
    'move the PC on
    IncAddress
    
End Sub
Public Sub PnM_WriteDataMemory(ByVal dataByte As Byte)

    Dim outval As Long
    outval = dataByte * 2
    
    SendCommand LOAD_DATA_FOR_DATA_MEMORY
    WriteBits outval
    'internally timed
    BeginProgramCycle
    delay10
    'move the PC on
    IncAddress

End Sub
Public Sub PnM_WriteConfiguration(ByVal ConfigLSB As Byte, ByVal ConfigMSB As Byte)
    
    ' set configuration mode
    PnM_EnterConfigurationMode
    ' seek to configuration memory
    IncAddress 7
    ' write the content
    PnM_WriteProgramMemory ConfigLSB, ConfigMSB
    
End Sub
Private Sub SendCommand(ByVal command As Byte)

    Dim intLoop As Integer
    Dim PORTC As Byte
    'largely JB code to send commands to the PIC
    For intLoop = 1 To 6
        PORTC = command And 1 Or 16
        Out Port1, (PORTC Or 2) ' also clk
        t = Timer 'intentional delay
        Out Port1, PORTC 'val plus clock low
       command = command \ 2
    Next
    
End Sub
Private Sub WriteBits(ByVal word As Long)
    
    Dim intLoop As Integer
    Dim PORTC As Byte
    'largely JB code to send word to the PIC
    For intLoop = 15 To 0 Step -1
        PORTC = word And 1 Or 16
        Out Port1, (PORTC Or 2) ' also clk
        t = Timer 'intentional delay
        Out Port1, PORTC 'val plus clock low
        word = word \ 2
    Next
    
End Sub
Public Sub delay10()
    
    'JB code to wait long delay
    d = longdelay
    While d > 0
        t = Int(Timer)
        d = d - 1
    Wend
    
End Sub
Private Sub delay1()

    'JB code for small delay
    t = Int(Timer)
    
End Sub
Private Sub BeginProgramCycle()

    SendCommand BEGIN_PROGRAMMING
    delay10 'this was delay4
    
End Sub
Private Sub IncAddress(Optional ByVal count As Integer = 1)
    
    Dim X As Integer
    'increment the PC by count
    For X = 1 To count
        SendCommand INCREMENT_ADDRESS
    Next
    
End Sub
Private Sub SetConfiguration()

    SendCommand LOAD_CONFIGURATION
    WriteBits 0
    
End Sub
Public Sub BulkEraseProgram()

    SendCommand BULK_ERASE_PROGRAM_MEMORY
    delay10
    
End Sub
Public Sub ChipErase()

    SendCommand CHIP_ERASE
    delay10
    
End Sub
Private Sub BulkEraseEEPROM()

    SendCommand BULK_ERASE_DATA_MEMORY
    delay10
    
End Sub
Public Sub PnM_ReadOscCal(ByVal oscAddr As Integer, ByRef OscLSB As Byte, ByRef OscMSB As Byte)
    
    ' seek to calibration memory
    IncAddress oscAddr
    ReadProgramMemory OscLSB, OscMSB
    
End Sub
Public Sub PnM_RestoreOscCal(ByVal oscAddr As Integer, ByRef OscLSB As Byte, ByRef OscMSB As Byte)
    
    ' seek to calibration memory
    IncAddress oscAddr
    PnM_WriteProgramMemory OscLSB, OscMSB
    
End Sub
Private Sub ReadProgramMemory(ByRef lobyte As Byte, ByRef hibyte As Byte)
    
    Dim word As Double
    word = 0
    SendCommand READ_DATA_FROM_PROGRAM_MEMORY
    'read program word
    ReadBits word
    'split into low and high  bytes
    hibyte = Int(word / 256)
    lobyte = word - (hibyte * 256)
    'move the PC on
    IncAddress
    
End Sub
Private Sub ReadBits(ByRef word As Double)
    
    Dim X As Integer
    Dim E As Integer
    'JB code to read program word
    For X = 15 To 0 Step -1
        word = word \ 2
        Out Port1, (16 Or 2): ' 12V on, clk high
        Call delay1
        Out Port1, 16 ' 12V on, clk low
        Call delay1
        E = Inp(Port2)  ' get bit
        E = E And 64  ' mask it out
        If E <> 0 Then word = word Or 32768
    Next
    word = word \ 2
    
End Sub
Public Sub PnM_ReadConfiguration(ByRef configL As Byte, ByRef configH As Byte)
    
    ' set configuration mode
    PnM_EnterConfigurationMode
    ' seek to configuration memory
    IncAddress 7
    ' read the content
    ReadProgramMemory configL, configH
    
End Sub
Public Function PnM_ReadPICType() As Integer
    
    Dim idL As Byte
    Dim idH As Byte
    
    PnM_EnterProgramMode
    ' set configuration mode
    PnM_EnterConfigurationMode
    ' seek to device ID memory
    IncAddress 6
    ' read the content
    ReadProgramMemory idL, idH
    PnM_ExitProgramMode
    'set the return value
    PnM_ReadPICType = idL + (256 * idH)
    
End Function

